<?php
/**
 * Dino Browser - Settings Sync API
 * Handles user settings synchronization
 */

require_once 'config.php';

$input = getInput();
$action = $input['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'get':
        handleGet();
        break;
    case 'sync':
        handleSync($input);
        break;
    default:
        jsonResponse(['success' => false, 'error' => 'Invalid action'], 400);
}

/**
 * Get settings for user
 */
function handleGet() {
    $userId = sanitize($_GET['user_id'] ?? '');
    
    if (empty($userId)) {
        jsonResponse(['success' => false, 'error' => 'Missing user_id'], 400);
    }
    
    $pdo = getConnection();
    
    try {
        $stmt = $pdo->prepare("SELECT settings_json, updated_at FROM settings WHERE user_id = ?");
        $stmt->execute([$userId]);
        $row = $stmt->fetch();
        
        if ($row) {
            $settings = json_decode($row['settings_json'], true);
            
            logSync($pdo, $userId, 'settings', 1, 'download', 'success');
            
            jsonResponse([
                'success' => true,
                'settings' => $settings,
                'updatedAt' => $row['updated_at']
            ]);
        } else {
            jsonResponse([
                'success' => true,
                'settings' => null,
                'message' => 'No settings found'
            ]);
        }
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Failed to fetch settings'], 500);
    }
}

/**
 * Sync settings from client
 */
function handleSync($input) {
    if (!validateRequired($input, ['user_id', 'settings'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $settings = $input['settings'];
    
    if (!is_array($settings)) {
        jsonResponse(['success' => false, 'error' => 'Invalid settings format'], 400);
    }
    
    // Remove sensitive/local-only settings before storing
    unset($settings['downloadPath']); // Local path
    
    $settingsJson = json_encode($settings, JSON_UNESCAPED_UNICODE);
    
    $pdo = getConnection();
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO settings (user_id, settings_json)
            VALUES (?, ?)
            ON DUPLICATE KEY UPDATE
                settings_json = VALUES(settings_json),
                updated_at = NOW()
        ");
        $stmt->execute([$userId, $settingsJson]);
        
        logSync($pdo, $userId, 'settings', 1, 'upload', 'success');
        
        jsonResponse([
            'success' => true,
            'message' => 'Settings synced'
        ]);
        
    } catch (PDOException $e) {
        logSync($pdo, $userId, 'settings', 0, 'upload', 'failed', $e->getMessage());
        jsonResponse(['success' => false, 'error' => 'Sync failed'], 500);
    }
}
