<?php
/**
 * Sync Progress API
 * Receives real-time student progress when online
 */

require_once '../config/database.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

$data = json_decode(file_get_contents('php://input'), true);

$session_token = $data['session_token'] ?? '';
$student_email = $data['student_email'] ?? '';
$student_name = $data['student_name'] ?? '';
$roll_no = $data['roll_no'] ?? '';
$exam_code = $data['exam_code'] ?? '';
$answers = $data['answers'] ?? [];
$time_remaining = $data['time_remaining'] ?? 0;
$current_question = $data['current_question'] ?? 0;
$total_questions = $data['total_questions'] ?? 0;

if (empty($session_token)) {
    http_response_code(400);
    die(json_encode(['error' => 'Session token required']));
}

try {
    // Check if progress record exists
    $stmt = $pdo->prepare("SELECT id FROM exam_progress WHERE session_token = ?");
    $stmt->execute([$session_token]);
    $existing = $stmt->fetch();

    // Calculate answered count
    $answered_count = 0;
    foreach ($answers as $qid => $choices) {
        if (is_array($choices) && count($choices) > 0) {
            $answered_count++;
        }
    }

    $answers_json = json_encode($answers);
    $now = date('Y-m-d H:i:s');

    if ($existing) {
        // Update existing progress
        $stmt = $pdo->prepare("
            UPDATE exam_progress SET 
                answers_json = ?,
                time_remaining = ?,
                current_question = ?,
                answered_count = ?,
                last_sync = ?
            WHERE session_token = ?
        ");
        $stmt->execute([
            $answers_json,
            $time_remaining,
            $current_question,
            $answered_count,
            $now,
            $session_token
        ]);
    } else {
        // Insert new progress record
        $stmt = $pdo->prepare("
            INSERT INTO exam_progress (
                session_token, student_email, student_name, roll_no, exam_code,
                answers_json, time_remaining, current_question, total_questions,
                answered_count, last_sync, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $session_token,
            $student_email,
            $student_name,
            $roll_no,
            $exam_code,
            $answers_json,
            $time_remaining,
            $current_question,
            $total_questions,
            $answered_count,
            $now,
            $now
        ]);
    }

    // Also update time in active_sessions
    $stmt = $pdo->prepare("
        UPDATE active_sessions SET 
            time_left_seconds = ?,
            current_question_index = ?
        WHERE session_token = ?
    ");
    $stmt->execute([$time_remaining, $current_question, $session_token]);

    echo json_encode([
        'success' => true,
        'message' => 'Progress synced',
        'synced_at' => $now
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error', 'details' => $e->getMessage()]);
}
?>
