<?php
/**
 * Dino Browser - Summaries Sync API
 * Handles AI summary storage and retrieval
 */

require_once 'config.php';

$input = getInput();
$action = $input['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'get':
        handleGet();
        break;
    case 'sync':
        handleSync($input);
        break;
    case 'delete':
        handleDelete($input);
        break;
    default:
        jsonResponse(['success' => false, 'error' => 'Invalid action'], 400);
}

/**
 * Get all summaries for user
 */
function handleGet() {
    $userId = sanitize($_GET['user_id'] ?? '');
    $limit = intval($_GET['limit'] ?? 100);
    $onlyYouTube = isset($_GET['youtube']) && $_GET['youtube'] === '1';
    
    if (empty($userId)) {
        jsonResponse(['success' => false, 'error' => 'Missing user_id'], 400);
    }
    
    $limit = min($limit, 500);
    
    $pdo = getConnection();
    
    try {
        $sql = "
            SELECT id, url, page_title, summary, is_youtube, created_at
            FROM summaries 
            WHERE user_id = ?
        ";
        $params = [$userId];
        
        if ($onlyYouTube) {
            $sql .= " AND is_youtube = 1";
        }
        
        $sql .= " ORDER BY created_at DESC LIMIT ?";
        $params[] = $limit;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $summaries = $stmt->fetchAll();
        
        // Convert is_youtube to boolean
        foreach ($summaries as &$summary) {
            $summary['isYouTube'] = (bool) $summary['is_youtube'];
            $summary['pageTitle'] = $summary['page_title'];
            $summary['createdAt'] = $summary['created_at'];
            unset($summary['is_youtube'], $summary['page_title'], $summary['created_at']);
        }
        
        logSync($pdo, $userId, 'summaries', count($summaries), 'download', 'success');
        
        jsonResponse([
            'success' => true,
            'summaries' => $summaries
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Failed to fetch summaries'], 500);
    }
}

/**
 * Sync summaries from client
 */
function handleSync($input) {
    if (!validateRequired($input, ['user_id', 'summaries'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $summaries = $input['summaries'];
    
    $pdo = getConnection();
    
    try {
        $pdo->beginTransaction();
        $syncedCount = 0;
        
        foreach ($summaries as $summary) {
            $summaryId = sanitize($summary['id'] ?? '');
            if (empty($summaryId)) continue;
            
            $stmt = $pdo->prepare("
                INSERT INTO summaries (id, user_id, url, page_title, summary, is_youtube, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    page_title = VALUES(page_title),
                    summary = VALUES(summary)
            ");
            
            $stmt->execute([
                $summaryId,
                $userId,
                $summary['url'] ?? '',
                sanitize($summary['pageTitle'] ?? ''),
                $summary['summary'] ?? '',
                ($summary['isYouTube'] ?? false) ? 1 : 0,
                $summary['createdAt'] ?? date('Y-m-d H:i:s')
            ]);
            
            $syncedCount++;
        }
        
        $pdo->commit();
        
        logSync($pdo, $userId, 'summaries', $syncedCount, 'upload', 'success');
        
        jsonResponse([
            'success' => true,
            'synced' => $syncedCount
        ]);
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        logSync($pdo, $userId, 'summaries', 0, 'upload', 'failed', $e->getMessage());
        jsonResponse(['success' => false, 'error' => 'Sync failed'], 500);
    }
}

/**
 * Delete a summary
 */
function handleDelete($input) {
    if (!validateRequired($input, ['user_id', 'summary_id'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $summaryId = sanitize($input['summary_id']);
    
    $pdo = getConnection();
    
    try {
        $stmt = $pdo->prepare("DELETE FROM summaries WHERE id = ? AND user_id = ?");
        $stmt->execute([$summaryId, $userId]);
        
        jsonResponse([
            'success' => true,
            'deleted' => $stmt->rowCount() > 0
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Delete failed'], 500);
    }
}
