<?php
require_once '../config/database.php';
require_once 'verify-browser.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

$data = json_decode(file_get_contents('php://input'), true);

// Support both student_code and email-based authentication
$student_code = $data['student_code'] ?? '';
$student_email = $data['student_email'] ?? '';
$student_name = $data['student_name'] ?? '';
$roll_no = $data['roll_no'] ?? '';
$exam_code = $data['exam_code'] ?? '';

if (empty($exam_code)) {
    http_response_code(400);
    die(json_encode(['success' => false, 'message' => 'Exam code is required']));
}

if (empty($student_code) && empty($student_email)) {
    http_response_code(400);
    die(json_encode(['success' => false, 'message' => 'Student code or email is required']));
}

try {
    // Get learner - try by email first, then by student_code
    $learner = null;
    
    if (!empty($student_email)) {
        // Using correct column name: email_address
        $stmt = $pdo->prepare("SELECT learner_id, full_name, email_address FROM learners WHERE email_address = ?");
        $stmt->execute([$student_email]);
        $learner = $stmt->fetch();
        
        // If not found by email, create a new learner (auto-register from DINO browser)
        if (!$learner && !empty($student_name)) {
            // Generate a unique student_code
            $auto_code = 'BGNU_' . substr(md5($student_email), 0, 8);
            // Using correct column names: full_name, email_address, account_type
            $stmt = $pdo->prepare("INSERT INTO learners (student_code, full_name, email_address, password_hash, account_type) VALUES (?, ?, ?, ?, 'student')");
            $stmt->execute([$auto_code, $student_name, $student_email, password_hash('dino_temp_' . time(), PASSWORD_DEFAULT)]);
            $learner_id = $pdo->lastInsertId();
            $learner = [
                'learner_id' => $learner_id,
                'full_name' => $student_name,
                'email_address' => $student_email
            ];
        }
    }
    
    if (!$learner && !empty($student_code)) {
        $stmt = $pdo->prepare("SELECT learner_id, full_name, email_address FROM learners WHERE student_code = ?");
        $stmt->execute([$student_code]);
        $learner = $stmt->fetch();
    }

    if (!$learner) {
        http_response_code(404);
        die(json_encode(['success' => false, 'message' => 'Student not found. Please ensure your account is registered.']));
    }

    // Get assessment
    $stmt = $pdo->prepare("SELECT * FROM assessments WHERE exam_code = ? AND assessment_status = 'active'");
    $stmt->execute([$exam_code]);
    $assessment = $stmt->fetch();

    if (!$assessment) {
        http_response_code(404);
        die(json_encode(['success' => false, 'message' => 'Exam not found or not active']));
    }

    // Check if already started
    $stmt = $pdo->prepare("SELECT * FROM active_sessions WHERE learner_id = ? AND assessment_id = ? AND session_status = 'active'");
    $stmt->execute([$learner['learner_id'], $assessment['assessment_id']]);
    $existing = $stmt->fetch();

    if ($existing) {
        // Get questions for resumed session
        $stmt = $pdo->prepare("
            SELECT question_id, question_order, question_prompt, 
                   choice_a, choice_b, choice_c, choice_d, point_value
            FROM question_bank 
            WHERE assessment_id = ? 
            ORDER BY question_order ASC
        ");
        $stmt->execute([$assessment['assessment_id']]);
        $questions = $stmt->fetchAll();
        
        // Resume session
        echo json_encode([
            'success' => true,
            'resumed' => true,
            'session_token' => $existing['session_token'],
            'current_question' => $existing['current_question_index'],
            'time_remaining_seconds' => $existing['time_left_seconds'],
            'exam' => [
                'exam_code' => $assessment['exam_code'],
                'title' => $assessment['assessment_title'],
                'duration_minutes' => $assessment['total_duration_mins'],
                'total_questions' => $assessment['question_count'],
                'seconds_per_question' => 60
            ],
            'questions' => $questions
        ]);
        exit;
    }

    // Create new session
    $session_token = bin2hex(random_bytes(32));
    $time_seconds = $assessment['total_duration_mins'] * 60;
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';

    $stmt = $pdo->prepare("
        INSERT INTO active_sessions (
            session_token, learner_id, assessment_id, time_left_seconds,
            session_started_at, ip_address
        ) VALUES (?, ?, ?, ?, NOW(), ?)
    ");
    $stmt->execute([
        $session_token,
        $learner['learner_id'],
        $assessment['assessment_id'],
        $time_seconds,
        $ip_address
    ]);

    // Get questions
    $stmt = $pdo->prepare("
        SELECT question_id, question_order, question_prompt, 
               choice_a, choice_b, choice_c, choice_d, point_value
        FROM question_bank 
        WHERE assessment_id = ? 
        ORDER BY question_order ASC
    ");
    $stmt->execute([$assessment['assessment_id']]);
    $questions = $stmt->fetchAll();

    echo json_encode([
        'success' => true,
        'session_token' => $session_token,
        'exam' => [
            'exam_code' => $assessment['exam_code'],
            'title' => $assessment['assessment_title'],
            'duration_minutes' => $assessment['total_duration_mins'],
            'total_questions' => $assessment['question_count'],
            'seconds_per_question' => 60
        ],
        'questions' => $questions,
        'started_at' => date('Y-m-d H:i:s')
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
}
?>
