/* ==========================================
   BGNU Exam Portal - JavaScript Logic
   Offline-First Exam System with Anti-Cheat
   ========================================== */

const API_BASE = 'https://bilalcode.site/exammode/api';

// State
let examState = {
    sessionToken: null,
    rollNo: null,
    studentEmail: null,
    studentName: null,
    questions: [],
    answers: {},
    currentIndex: 0,
    timeRemaining: 0,
    timerInterval: null,
    isOnline: navigator.onLine,
    examStarted: false,
    isTerminated: false,
    violations: []
};

// DOM Elements
const elements = {
    preExamScreen: document.getElementById('pre-exam-screen'),
    examScreen: document.getElementById('exam-screen'),
    resultScreen: document.getElementById('result-screen'),
    loadDataBtn: document.getElementById('load-data-btn'),
    loadStatus: document.getElementById('load-status'),
    rollNoInput: document.getElementById('roll-no'),
    studentEmailInput: document.getElementById('student-email'),
    studentNameInput: document.getElementById('student-name'),
    examInfo: document.getElementById('exam-info'),
    examTitle: document.getElementById('exam-title'),
    examDuration: document.getElementById('exam-duration'),
    examQuestions: document.getElementById('exam-questions'),
    timeDisplay: document.getElementById('time-display'),
    connectionStatus: document.getElementById('connection-status'),
    progressBar: document.getElementById('progress-bar'),
    progressText: document.getElementById('progress-text'),
    questionNumber: document.getElementById('question-number'),
    questionPrompt: document.getElementById('question-prompt'),
    optionsContainer: document.getElementById('options-container'),
    prevBtn: document.getElementById('prev-btn'),
    nextBtn: document.getElementById('next-btn'),
    questionDots: document.getElementById('question-dots'),
    submitBtn: document.getElementById('submit-btn'),
    timer: document.getElementById('timer'),
    resultIcon: document.getElementById('result-icon'),
    resultTitle: document.getElementById('result-title'),
    resultMessage: document.getElementById('result-message'),
    resultStats: document.getElementById('result-stats')
};

// ==================== INITIALIZATION ====================

document.addEventListener('DOMContentLoaded', () => {
    // Check for saved exam session
    loadSavedSession();

    // Online/Offline detection
    window.addEventListener('online', () => updateOnlineStatus(true));
    window.addEventListener('offline', () => updateOnlineStatus(false));
    updateOnlineStatus(navigator.onLine);

    // Event Listeners
    elements.loadDataBtn.addEventListener('click', loadExamData);
    elements.prevBtn.addEventListener('click', () => navigateQuestion(-1));
    elements.nextBtn.addEventListener('click', () => navigateQuestion(1));
    elements.submitBtn.addEventListener('click', submitExam);

    // ========== ANTI-CHEAT: TEMPORARILY DISABLED FOR WINDOWS TESTING ==========
    // Uncomment after fixing Windows crash
    // document.addEventListener('visibilitychange', handleVisibilityChange);
    // window.addEventListener('blur', handleWindowBlur);
    // window.addEventListener('focus', handleWindowFocus);

    // Disable right-click (keeping this as it's harmless)
    document.addEventListener('contextmenu', (e) => {
        if (examState.examStarted && !examState.isTerminated) {
            e.preventDefault();
        }
    });

    // Disable keyboard shortcuts
    // document.addEventListener('keydown', handleKeyDown);

    // Notify parent (Flutter) - disabled temporarily
    // notifyParent('exam_page_loaded', {});
});

// ==================== ANTI-CHEAT FUNCTIONS ====================

function handleVisibilityChange() {
    if (!examState.examStarted || examState.isTerminated) return;

    if (document.hidden) {
        // Student switched tabs or minimized
        terminateExam('tab_switch', 'You switched away from the exam. Your exam has been terminated.');
    }
}

function handleWindowBlur() {
    if (!examState.examStarted || examState.isTerminated) return;
    // Window lost focus (Alt+Tab, clicking outside, etc.)
    terminateExam('window_blur', 'You left the exam window. Your exam has been terminated.');
}

function handleWindowFocus() {
    // Can be used to resume if we want to allow one warning
}

function handleKeyDown(e) {
    if (!examState.examStarted || examState.isTerminated) return;

    // Block Alt+Tab, Alt+F4, Ctrl+Tab, F11, etc.
    const blockedKeys = ['Tab', 'F4', 'F5', 'F11', 'Escape'];
    const isAltCombo = e.altKey && blockedKeys.includes(e.key);
    const isCtrlCombo = e.ctrlKey && ['Tab', 'w', 'W', 'n', 'N', 't', 'T'].includes(e.key);

    if (isAltCombo || isCtrlCombo) {
        e.preventDefault();
        recordViolation('blocked_key', `Attempted: ${e.ctrlKey ? 'Ctrl' : 'Alt'}+${e.key}`);
        return false;
    }

    // Block F5 refresh
    if (e.key === 'F5') {
        e.preventDefault();
        recordViolation('refresh_attempt', 'Attempted page refresh');
        return false;
    }
}

function recordViolation(type, description) {
    const violation = {
        type: type,
        description: description,
        timestamp: new Date().toISOString()
    };
    examState.violations.push(violation);

    // Send to server
    if (examState.isOnline && examState.sessionToken) {
        fetch(`${API_BASE}/record-violation.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                session_token: examState.sessionToken,
                violation_type: type,
                description: description
            })
        }).catch(e => console.warn('Failed to record violation:', e));
    }
}

function terminateExam(reason, message) {
    if (examState.isTerminated) return;

    examState.isTerminated = true;
    clearInterval(examState.timerInterval);

    // Record the violation
    recordViolation(reason, message);

    // Notify server
    if (examState.isOnline && examState.sessionToken) {
        fetch(`${API_BASE}/terminate-exam.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                session_token: examState.sessionToken,
                reason: reason,
                message: message
            })
        }).catch(e => console.warn('Failed to terminate on server:', e));
    }

    // Show termination screen
    showTerminatedScreen(message);

    // Clear local session
    clearSession();

    // Notify parent (Flutter)
    notifyParent('exam_terminated', { reason: reason, message: message });
}

function showTerminatedScreen(message) {
    elements.preExamScreen.classList.add('hidden');
    elements.examScreen.classList.add('hidden');
    elements.resultScreen.classList.remove('hidden');

    elements.resultIcon.textContent = '🚫';
    elements.resultTitle.textContent = 'Exam Terminated';
    elements.resultMessage.textContent = message;
    elements.resultStats.innerHTML = `
        <div class="stat-item" style="grid-column: 1/-1;">
            <div class="stat-value" style="color: #ef4444;">CHEATING DETECTED</div>
            <div class="stat-label">Your exam has been invalidated</div>
        </div>
    `;
}

// ==================== PARENT COMMUNICATION (Flutter) ====================

function notifyParent(event, data) {
    // Send message to Flutter WebView
    try {
        if (window.flutter_inappwebview) {
            window.flutter_inappwebview.callHandler('examEvent', { event, data });
        }
    } catch (e) {
        console.log('Parent notification:', event, data);
    }
}

// ==================== ONLINE STATUS ====================

function updateOnlineStatus(isOnline) {
    examState.isOnline = isOnline;
    const statusEl = elements.connectionStatus;
    const dot = statusEl.querySelector('.status-dot');
    const text = statusEl.querySelector('span:last-child');

    if (isOnline) {
        dot.classList.add('online');
        dot.classList.remove('offline');
        text.textContent = 'Online';
        elements.submitBtn.disabled = false;

        // Sync pending answers when back online
        syncPendingAnswers();
    } else {
        dot.classList.remove('online');
        dot.classList.add('offline');
        text.textContent = 'Offline';
    }
}

// ==================== LOAD EXAM DATA ====================

async function loadExamData() {
    const rollNo = elements.rollNoInput.value.trim();
    const studentEmail = elements.studentEmailInput.value.trim();
    const studentName = elements.studentNameInput.value.trim();

    if (!rollNo || !studentEmail || !studentName) {
        showStatus('Please fill in all fields', 'error');
        return;
    }

    elements.loadDataBtn.disabled = true;
    showStatus('Loading exam data...', 'loading');

    try {
        const response = await fetch(`${API_BASE}/start-exam.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                roll_no: rollNo,
                student_email: studentEmail,
                student_name: studentName,
                exam_code: 'TEST2026' // Default exam for now
            })
        });

        const data = await response.json();

        if (!data.success) {
            throw new Error(data.message || 'Failed to load exam');
        }

        // Store exam data
        examState.sessionToken = data.session_token;
        examState.rollNo = rollNo;
        examState.studentEmail = studentEmail;
        examState.studentName = studentName;
        examState.questions = data.questions;
        examState.timeRemaining = data.exam.duration_minutes * 60;

        // Load any previous answers if resuming
        if (data.resumed) {
            examState.timeRemaining = data.time_remaining_seconds;
            loadSavedAnswers();
        }

        // Save to localStorage
        saveSession();

        showStatus('Exam loaded successfully!', 'success');

        // Show exam info
        elements.examInfo.style.display = 'block';
        elements.examTitle.textContent = data.exam.title;
        elements.examDuration.textContent = `${data.exam.duration_minutes} minutes`;
        elements.examQuestions.textContent = `${data.exam.total_questions} questions`;

        // Notify parent to enable lockdown
        notifyParent('exam_starting', {
            sessionToken: data.session_token,
            rollNo: rollNo
        });

        // Transition to exam after brief delay
        setTimeout(() => {
            startExam();
        }, 1000);

    } catch (error) {
        console.error('Load error:', error);

        // Try to load from localStorage if offline
        if (!navigator.onLine) {
            const saved = loadSavedSession();
            if (saved) {
                showStatus('Loaded from offline storage', 'success');
                setTimeout(() => startExam(), 1000);
                return;
            }
        }

        showStatus(error.message || 'Failed to load exam. Check connection.', 'error');
        elements.loadDataBtn.disabled = false;
    }
}

// ==================== EXAM FLOW ====================

function startExam() {
    examState.examStarted = true;

    elements.preExamScreen.classList.add('hidden');
    elements.examScreen.classList.remove('hidden');
    elements.submitBtn.classList.remove('hidden');

    // Notify parent that exam has officially started
    notifyParent('exam_started', {
        sessionToken: examState.sessionToken,
        rollNo: examState.rollNo,
        studentName: examState.studentName
    });

    // Start timer
    startTimer();

    // Render first question
    renderQuestion(0);
    renderQuestionDots();
    updateProgress();
}

function renderQuestion(index) {
    if (examState.isTerminated) return;

    const question = examState.questions[index];
    if (!question) return;

    examState.currentIndex = index;

    // Update question display
    elements.questionNumber.textContent = `Question ${index + 1}`;
    elements.questionPrompt.textContent = question.question_prompt;

    // Render options
    const options = [
        { letter: 'A', text: question.choice_a },
        { letter: 'B', text: question.choice_b },
        { letter: 'C', text: question.choice_c },
        { letter: 'D', text: question.choice_d }
    ];

    const savedAnswers = examState.answers[question.question_id] || [];

    elements.optionsContainer.innerHTML = options.map(opt => `
        <div class="option-item ${savedAnswers.includes(opt.letter) ? 'selected' : ''}" 
             data-letter="${opt.letter}" onclick="selectOption('${opt.letter}')">
            <div class="option-checkbox"></div>
            <div class="option-letter">${opt.letter}</div>
            <div class="option-text">${opt.text}</div>
        </div>
    `).join('');

    // Update navigation buttons
    elements.prevBtn.disabled = index === 0;
    elements.nextBtn.textContent = index === examState.questions.length - 1 ? 'Finish' : 'Next →';

    // Update dots
    updateQuestionDots();
}

function selectOption(letter) {
    if (examState.isTerminated) return;

    const question = examState.questions[examState.currentIndex];
    const questionId = question.question_id;

    // Toggle selection (multi-select support)
    if (!examState.answers[questionId]) {
        examState.answers[questionId] = [];
    }

    const idx = examState.answers[questionId].indexOf(letter);
    if (idx > -1) {
        examState.answers[questionId].splice(idx, 1);
    } else {
        examState.answers[questionId].push(letter);
    }

    // Save to localStorage immediately
    saveAnswers();

    // Sync to server if online
    if (examState.isOnline) {
        syncAnswer(questionId, examState.answers[questionId]);
    }

    // Re-render to show selection
    renderQuestion(examState.currentIndex);
    updateProgress();
}

function navigateQuestion(direction) {
    if (examState.isTerminated) return;

    const newIndex = examState.currentIndex + direction;

    if (newIndex >= 0 && newIndex < examState.questions.length) {
        renderQuestion(newIndex);
    } else if (newIndex >= examState.questions.length) {
        // Show submit confirmation
        elements.submitBtn.scrollIntoView({ behavior: 'smooth' });
    }
}

// ==================== QUESTION DOTS ====================

function renderQuestionDots() {
    elements.questionDots.innerHTML = examState.questions.map((q, i) => `
        <div class="dot" data-index="${i}" onclick="jumpToQuestion(${i})"></div>
    `).join('');
}

function updateQuestionDots() {
    const dots = elements.questionDots.querySelectorAll('.dot');
    dots.forEach((dot, i) => {
        const question = examState.questions[i];
        const isAnswered = examState.answers[question.question_id]?.length > 0;

        dot.classList.toggle('answered', isAnswered);
        dot.classList.toggle('current', i === examState.currentIndex);
    });
}

function jumpToQuestion(index) {
    if (examState.isTerminated) return;
    renderQuestion(index);
}

// ==================== PROGRESS ====================

function updateProgress() {
    const answered = Object.values(examState.answers).filter(a => a.length > 0).length;
    const total = examState.questions.length;
    const percent = (answered / total) * 100;

    elements.progressBar.style.setProperty('--progress', `${percent}%`);
    elements.progressText.textContent = `${answered} / ${total}`;
}

// ==================== TIMER ====================

function startTimer() {
    updateTimerDisplay();

    examState.timerInterval = setInterval(() => {
        if (examState.isTerminated) {
            clearInterval(examState.timerInterval);
            return;
        }

        examState.timeRemaining--;
        updateTimerDisplay();

        // Save time to localStorage every 10 seconds
        if (examState.timeRemaining % 10 === 0) {
            saveSession();
        }

        // Auto-submit when time runs out
        if (examState.timeRemaining <= 0) {
            clearInterval(examState.timerInterval);
            submitExam(true);
        }

        // Warning at 5 minutes
        if (examState.timeRemaining === 300) {
            elements.timer.style.color = '#f59e0b';
        }

        // Critical at 1 minute
        if (examState.timeRemaining === 60) {
            elements.timer.style.color = '#ef4444';
        }
    }, 1000);
}

function updateTimerDisplay() {
    const minutes = Math.floor(examState.timeRemaining / 60);
    const seconds = examState.timeRemaining % 60;
    elements.timeDisplay.textContent = `${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
}

// ==================== LOCAL STORAGE ====================

function saveSession() {
    const sessionData = {
        sessionToken: examState.sessionToken,
        rollNo: examState.rollNo,
        studentEmail: examState.studentEmail,
        studentName: examState.studentName,
        questions: examState.questions,
        timeRemaining: examState.timeRemaining,
        examStarted: examState.examStarted,
        savedAt: new Date().toISOString()
    };
    localStorage.setItem('bgnu_exam_session', JSON.stringify(sessionData));
}

function saveAnswers() {
    localStorage.setItem('bgnu_exam_answers', JSON.stringify(examState.answers));
}

function loadSavedSession() {
    try {
        const sessionData = localStorage.getItem('bgnu_exam_session');
        const answersData = localStorage.getItem('bgnu_exam_answers');

        if (sessionData) {
            const session = JSON.parse(sessionData);
            examState.sessionToken = session.sessionToken;
            examState.rollNo = session.rollNo;
            examState.studentEmail = session.studentEmail;
            examState.studentName = session.studentName;
            examState.questions = session.questions;
            examState.timeRemaining = session.timeRemaining;

            // Pre-fill form
            if (elements.rollNoInput) elements.rollNoInput.value = session.rollNo || '';
            if (elements.studentEmailInput) elements.studentEmailInput.value = session.studentEmail || '';
            if (elements.studentNameInput) elements.studentNameInput.value = session.studentName || '';
        }

        if (answersData) {
            examState.answers = JSON.parse(answersData);
        }

        return sessionData !== null && examState.questions && examState.questions.length > 0;
    } catch (e) {
        console.error('Failed to load saved session:', e);
        return false;
    }
}

function loadSavedAnswers() {
    const saved = localStorage.getItem('bgnu_exam_answers');
    if (saved) {
        examState.answers = JSON.parse(saved);
    }
}

function clearSession() {
    localStorage.removeItem('bgnu_exam_session');
    localStorage.removeItem('bgnu_exam_answers');
}

// ==================== SYNC TO SERVER ====================

async function syncAnswer(questionId, selectedChoices) {
    if (!examState.isOnline || examState.isTerminated) return;

    try {
        // Sync individual answer
        await fetch(`${API_BASE}/submit-answer.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                session_token: examState.sessionToken,
                question_id: questionId,
                selected_choice: selectedChoices.join(','),
                response_time_seconds: 0
            })
        });

        // Also sync progress
        await fetch(`${API_BASE}/sync-progress.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                session_token: examState.sessionToken,
                student_email: examState.studentEmail,
                student_name: examState.studentName,
                roll_no: examState.rollNo,
                exam_code: 'TEST2026',
                answers: examState.answers,
                time_remaining: examState.timeRemaining,
                current_question: examState.currentIndex + 1,
                total_questions: examState.questions.length
            })
        });
    } catch (e) {
        console.warn('Sync failed, will retry when online:', e);
    }
}

async function syncPendingAnswers() {
    if (!examState.isOnline || !examState.sessionToken || examState.isTerminated) return;

    try {
        // Sync all answers
        for (const [qid, choices] of Object.entries(examState.answers)) {
            if (choices.length > 0) {
                await fetch(`${API_BASE}/submit-answer.php`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        session_token: examState.sessionToken,
                        question_id: parseInt(qid),
                        selected_choice: choices.join(','),
                        response_time_seconds: 0
                    })
                });
            }
        }
        console.log('All pending answers synced');
    } catch (e) {
        console.warn('Failed to sync pending answers:', e);
    }
}

// ==================== SUBMIT EXAM ====================

async function submitExam(autoSubmit = false) {
    if (examState.isTerminated) return;

    if (!examState.isOnline) {
        alert('You are offline. Please connect to the internet to submit your exam.');
        return;
    }

    if (!autoSubmit) {
        const answered = Object.values(examState.answers).filter(a => a.length > 0).length;
        const total = examState.questions.length;

        if (answered < total) {
            const confirm = window.confirm(`You have answered ${answered} of ${total} questions. Are you sure you want to submit?`);
            if (!confirm) return;
        }
    }

    clearInterval(examState.timerInterval);
    elements.submitBtn.disabled = true;
    elements.submitBtn.textContent = 'Submitting...';

    try {
        // First sync all answers
        await syncPendingAnswers();

        // Then submit exam
        const response = await fetch(`${API_BASE}/submit-exam.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                session_token: examState.sessionToken
            })
        });

        const data = await response.json();

        if (data.success) {
            showResult(data.result, autoSubmit);
            clearSession();
            notifyParent('exam_completed', { result: data.result });
        } else {
            throw new Error(data.error || 'Submission failed');
        }
    } catch (e) {
        console.error('Submit error:', e);
        alert('Failed to submit. Your answers are saved locally. Please try again.');
        elements.submitBtn.disabled = false;
        elements.submitBtn.textContent = '✅ Submit Exam';
    }
}

function showResult(result, wasAutoSubmit) {
    elements.examScreen.classList.add('hidden');
    elements.resultScreen.classList.remove('hidden');

    if (wasAutoSubmit) {
        elements.resultIcon.textContent = '⏰';
        elements.resultTitle.textContent = 'Time\'s Up!';
        elements.resultMessage.textContent = 'Your exam has been auto-submitted.';
    } else {
        elements.resultIcon.textContent = result.passed ? '🎉' : '📝';
        elements.resultTitle.textContent = result.passed ? 'Congratulations!' : 'Exam Submitted';
        elements.resultMessage.textContent = result.passed ? 'You passed the exam!' : 'Your answers have been recorded.';
    }

    elements.resultStats.innerHTML = `
        <div class="stat-item">
            <div class="stat-value">${result.score}</div>
            <div class="stat-label">Score</div>
        </div>
        <div class="stat-item">
            <div class="stat-value">${result.max_score}</div>
            <div class="stat-label">Max Score</div>
        </div>
        <div class="stat-item">
            <div class="stat-value">${result.percentage}%</div>
            <div class="stat-label">Percentage</div>
        </div>
    `;
}

// ==================== HELPERS ====================

function showStatus(message, type) {
    elements.loadStatus.textContent = message;
    elements.loadStatus.className = 'helper-text';

    if (type === 'error') {
        elements.loadStatus.style.color = '#ef4444';
    } else if (type === 'success') {
        elements.loadStatus.style.color = '#22c55e';
    } else {
        elements.loadStatus.style.color = '#a0a0b0';
    }
}

// Make functions available globally for onclick handlers
window.selectOption = selectOption;
window.jumpToQuestion = jumpToQuestion;
